const db = require('../models');
const Product = db.Product;

// 1. Tambah Produk Baru (Create)
exports.createProduct = async (req, res) => {
  try {
    const { name, description, price, stock } = req.body;
    
    // Cek apakah ada file gambar yang diupload
    let image_url = null;
    if (req.file) {
      // Kita simpan path-nya saja, misal: "uploads/123456.jpg"
      image_url = req.file.path; 
    }

    const product = await Product.create({
      name,
      description,
      price,
      stock,
      image_url
    });

    res.status(201).json({
      message: "Produk berhasil ditambahkan!",
      product
    });
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

// 2. Ambil Semua Produk (Read - Public)
exports.getAllProducts = async (req, res) => {
  try {
    // Ambil semua produk yang statusnya available
    const products = await Product.findAll({
      where: { is_available: true }
    });
    res.status(200).json(products);
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

// 3. Ambil Detail 1 Produk
exports.getProductById = async (req, res) => {
  try {
    const product = await db.Product.findByPk(req.params.id, {
      include: [{
        model: db.Review,
        include: [{ model: db.User, attributes: ['username'] }] // Tampilkan nama reviewer
      }]
    });
    if (!product) return res.status(404).json({ message: "Produk tidak ditemukan." });
    res.status(200).json(product);
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

// 4. Update Produk (Misal: Update stok atau harga)
exports.updateProduct = async (req, res) => {
  try {
    const id = req.params.id;
    // Jika ada file baru, update image_url juga
    if (req.file) {
      req.body.image_url = req.file.path;
    }

    const [updated] = await Product.update(req.body, {
      where: { id: id }
    });

    if (updated) {
      const updatedProduct = await Product.findByPk(id);
      res.status(200).json({ message: "Produk berhasil diupdate.", product: updatedProduct });
    } else {
      res.status(404).json({ message: "Produk tidak ditemukan." });
    }
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

// 5. Hapus Produk (Soft delete: set is_available = false)
// Kita jarang menghapus data permanen di toko agar riwayat pesanan tidak rusak
exports.deleteProduct = async (req, res) => {
  try {
    const id = req.params.id;
    await Product.update({ is_available: false }, {
      where: { id: id }
    });
    res.status(200).json({ message: "Produk dinonaktifkan (dihapus)." });
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};