const db = require('../models');
const Order = db.Order;
const OrderItem = db.OrderItem;
const Product = db.Product;
const User = db.User;
exports.createOrder = async (req, res) => {
  const t = await db.sequelize.transaction();
  try {
    // 1. PERBAIKAN: Ambil delivery_method & delivery_address dari body
    const { 
      items, 
      order_type, 
      customer_name, 
      payment_method, 
      delivery_method, 
      delivery_address 
    } = req.body;

    // 2. LOGIKA ID: Tentukan siapa yang membuat order
    // Jika Online, user_id diisi (agar muncul di history user).
    // Jika In-Person (Kasir), cashier_id diisi.
    let userId = null;
    let cashierId = null;

    if (order_type === 'online') {
      userId = req.userId; // ID User yang sedang login
    } else {
      cashierId = req.userId; // ID Kasir yang sedang login
    }

    // --- Cek Stok & Hitung Harga ---
    let totalPrice = 0;
    const orderItemsData = [];
    
    for (const item of items) {
       const product = await Product.findByPk(item.product_id, { transaction: t });
       
       if (!product) {
         throw new Error(`Produk dengan ID ${item.product_id} tidak ditemukan.`);
       }
       if (product.stock < item.quantity) {
         throw new Error(`Stok ${product.name} habis.`);
       }

       await product.update({ stock: product.stock - item.quantity }, { transaction: t });
       
       totalPrice += parseFloat(product.price) * item.quantity;
       orderItemsData.push({
         product_id: product.id, 
         quantity: item.quantity, 
         price_per_item: product.price
       });
    }

    // 3. Buat Order
    const order = await Order.create({
      user_id: userId,        // Terisi jika order online
      cashier_id: cashierId,  // Terisi jika order kasir
      total_price: totalPrice,
      status: 'paid',      // Default pending (bisa diubah kasir/payment gateway nanti)
      order_type: order_type || 'in-person',
      customer_name: customer_name || 'Pelanggan Umum',
      payment_method: payment_method || 'cash',
      // Masukkan data delivery
      delivery_method: delivery_method || 'pickup',
      delivery_address: delivery_address || null    
    }, { transaction: t });

    // 4. Jika ini Kasir (Bayar Tunai/Langsung), otomatis set ke 'paid'
    // (Opsional: User COD juga bisa dianggap pending/paid tergantung alur Anda)
    if (order_type === 'in-person') {
      await order.update({ status: 'paid' }, { transaction: t });
    }

    // Simpan Item
    const itemsWithOrderId = orderItemsData.map(item => ({ ...item, order_id: order.id }));
    await OrderItem.bulkCreate(itemsWithOrderId, { transaction: t });
    
    await t.commit();

    res.status(201).json({ message: "Pesanan berhasil!", order_id: order.id });

  } catch (err) {
    await t.rollback();
    console.error(err); // Penting untuk debugging di terminal
    res.status(500).json({ message: err.message || "Gagal memproses pesanan." });
  }
};

exports.getUserHistory = async (req, res) => {
  try {
    const orders = await db.Order.findAll({
      where: { user_id: req.userId }, // Khusus user yang login
      include: [{ model: db.Product }],
      order: [['created_at', 'DESC']]
    });
    res.status(200).json(orders);
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

// 2. BARU: Ambil Riwayat Transaksi Kasir (Hari Ini)
exports.getCashierHistory = async (req, res) => {
  try {
    // Ambil pesanan yang dibuat oleh kasir ini, diurutkan dari yang terbaru
    const orders = await Order.findAll({
      where: { cashier_id: req.userId },
      order: [['created_at', 'DESC']],
      limit: 50 // Batasi 50 transaksi terakhir agar ringan
    });
    res.status(200).json(orders);
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

// ... exports lainnya (getKitchenOrders, updateStatus)

// 2. Ambil Pesanan untuk DAPUR (Status: 'paid' atau 'processing')
exports.getKitchenOrders = async (req, res) => {
  try {
    const orders = await Order.findAll({
      where: {
        status: ['paid', 'processing'] // Hanya yang sudah dibayar atau sedang diproses
      },
      include: [
        {
          model: Product, // Sertakan detail produk (Roti apa aja?)
          through: { attributes: ['quantity'] } // Ambil jumlahnya
        },
        {
          model: User,
          as: 'cashier',
          attributes: ['username'] // Siapa kasirnya?
        }
      ],
      order: [['created_at', 'ASC']] // Yang lama di atas (FIFO)
    });

    res.status(200).json(orders);
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

// 3. Update Status Pesanan (Contoh: paid -> processing -> completed)
exports.updateStatus = async (req, res) => {
  try {
    const { status } = req.body;
    const order = await Order.findByPk(req.params.id);

    if (!order) return res.status(404).json({ message: "Pesanan tidak ditemukan." });

    order.status = status;
    await order.save();

    res.status(200).json({ message: `Status pesanan diperbarui menjadi ${status}` });
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};