const db = require('../models'); // Import dari /models/index.js
const User = db.User;
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');

require('dotenv').config();

// --- 1. Fungsi Registrasi (Hanya untuk 'user' online) ---
exports.register = async (req, res) => {
  try {
    const { username, password, full_name, email } = req.body;

    // Validasi input
    if (!username || !password || !full_name || !email) {
      return res.status(400).json({ message: 'Semua field harus diisi.' });
    }

    // 1. Cek apakah username atau email sudah ada
    const existingUser = await User.findOne({ 
      where: { [db.Sequelize.Op.or]: [{ username }, { email }] }
    });

    if (existingUser) {
      return res.status(409).json({ message: 'Username atau email sudah terdaftar.' });
    }

    // 2. Hash password
    const hashedPassword = await bcrypt.hash(password, 10);

    // 3. Buat user baru (default role 'user')
    const newUser = await User.create({
      username,
      password: hashedPassword,
      full_name,
      email,
      role: 'user' // Default role
    });

    res.status(201).json({ 
      message: 'Registrasi berhasil.',
      userId: newUser.id 
    });

  } catch (err) {
    res.status(500).json({ message: 'Terjadi kesalahan di server: ' + err.message });
  }
};

// --- 2. Fungsi Login (Untuk semua role) ---
exports.login = async (req, res) => {
  try {
    const { username, password } = req.body;

    // 1. Cari user berdasarkan username
    const user = await User.findOne({ where: { username } });
    
    if (!user) {
      return res.status(404).json({ message: 'Username tidak ditemukan.' });
    }

    // 2. Bandingkan password yang di-input dengan yang ada di database
    const isPasswordValid = await bcrypt.compare(password, user.password);

    if (!isPasswordValid) {
      return res.status(401).json({ message: 'Password salah.' });
    }

    // 3. Jika valid, buat JSON Web Token (JWT)
    const token = jwt.sign(
      { 
        id: user.id,
        role: user.role,
        username: user.username
      },
      process.env.JWT_SECRET,
      {
        expiresIn: '24h' // Token berlaku selama 24 jam
      }
    );

    // 4. Kirim token ke client
    res.status(200).json({
      message: 'Login berhasil.',
      user: {
        id: user.id,
        username: user.username,
        role: user.role
      },
      token: token
    });

  } catch (err) {
    res.status(500).json({ message: 'Terjadi kesalahan di server: ' + err.message });
  }
};